%
% This script reads large half hourly GXP data files produced by Gnash, and
% produces as output, the GXP projected peak demand to use as the base
% starting point for GXP forecasts. For example, if we have half-hourly
% demand data up till 2004, we will want a straight projection to use as a
% starting point for 2005. This 2005 starting point will then be inflated
% year by year, according to the GXP forecast, which is expressed only in
% terms of growth factors.
%
% Bruce Smith, 29th of August, 2005.
%
% Modified to calculate the diversity factors using the same
% methodology, and the power factors at those same points in time. These
% factors are used later to produce power flow models of future years. It
% is important to calculate diversity factors using the same methodology
% for determining the GXP peak as the base year peak, so that future
% load flow cases represent a realistic state of affairs during the island
% peaks.
%
% Bruce Smith, 30th of August, 2005.
%
% Added Island and Regional data handling November 2005  - Brian Kirtlan
% Last amended by Brian Kirtlan October 2007

clear all;

disp('Starting base ADMD calculations');
warning off MATLAB:divideByZero;

cd C:\ak\NatDemand2009\ADMD
graphdir = 'C:\ak\NatDemand2009\ADMD\figures';

% Get input data

readcontrol=1; % read gnash data and save it to binary form (use whenever gnash data updated)
%readcontrol=2; % read binary data (subsequent runs from scratch)
%readcontrol=3; % data already in memory (subsequent runs if modifying this script)

switch readcontrol
    case 1 % read from gnash and save to binary
        GnashFile='ADMDAllReactive09.dat';      % read in half hourly data from Gnash files
        Gnash2;
        ReactiveNames=GnashNames;
        ReactiveTitles=GnashTitles;
        ReactiveArray=GnashArray;
        save reactivedata ReactiveNames ReactiveTitles ReactiveArray
        clear GnashArray GnashTitles GnashNames;
        GnashFile='ADMDAllReal09.dat';             % read in half hourly data from Gnash files
        Gnash2;
        save realdata GnashNames GnashTitles GnashArray
    case 2 %read from binary
        load realdata;
        load reactivedata;
end

load 2009PSSE_GXP_NameMapping.mat; % BK - needed for region mapping
load 2009GXPForecasts.mat Regions % just want the region names
load 2009GXPForecasts.mat NewLoad % and any new load adjustments to peak
load 2009GXPForecasts.mat GXPExistingEmbeddedMW % and existing embedded generation to pass through to MakeIDEV 

% read specification for demand projection to use at each GXP. Comment out one or the other
% this version replicates the 2007 forecast (so needs year settings below
% to be from 2000-2006 and a base year of 2007
%[PTnames,PType,doMx(:,1),doMx(:,2),doMx(:,3),doMx(:,4),doMx(:,5),doMx(:,6),doMx(:,7),blankcol,PFixedMW]=textread('2009ProjectionType07.csv','%s%s%f%f%f%f%f%f%f%s%f','delimiter',',');

% this version uses 2000-2008 data so needs year settings below to be from
% 2000-2008 and a base year of 2009
[PTnames,PType,doMx(:,1),doMx(:,2),doMx(:,3),doMx(:,4),doMx(:,5),doMx(:,6),doMx(:,7),doMx(:,8),doMx(:,9),blankcol,PFixedMW]=textread('2009ProjectionType.csv','%s%s%f%f%f%f%f%f%f%f%f%s%f','delimiter',',');

DoGXPGraphs = 1; % Set to 1 to draw graphs for checking GXP peaks
DoRegionGraphs = 1; % Set to 1 to draw region and national graphs
WriteGraphsToFile = 1; % Set to 1 to write graphs to file if either graph switch above is set to 1
StepThroughGraphs = 0; % Set to 1 to step through graphs one-by-one as script runs (best done by writing graphs to file and using a viewing application if you want to step forwards and backwards) 

% Control costants

YearStt=2000;                           % start year for input data
YearStp=2008; %'08                          % end year
Npts=50;                                % Number of points to use in peak averaging each year
ESpts=20;                               % Number of points for Extreme Summer average
Spts=20;                                % Number of points for Summer average
Wpts=40;                                % Number of points for Winter average
BaseYear=2009;  %                         % the year for which a base projection is desired
ADMDYear=2007;  %'08 data mostly spoiled       % the year used to calculate diversity figures etc (generally the year that the reactive data covers)
TimeSummerRatingStarts = 15;            % first half hour in which summer line ratings start during summer (15 = half hour starting 7:00am)
TimeSummerRatingEnds = 40;              % last half hour in which summer line ratings end during summer (40 = half hour starting 8:30pm)

% Frequently used constants

NYears=YearStp-YearStt+1;
YearLength=365*48;                      % for convenience, all years to be same length
FirstDate=datenum(YearStt,1,1,0,0,0);   % end points for graphing against datenums
LastDate=datenum(BaseYear+1,1,1,0,0,0);
YearList=YearStt:YearStp;               % the 'x' index for fitting to yearly data
PlotList=FirstDate:LastDate;            % the 'x' index for daily data (only used for graphing)

idx=strmatch('ADMDLoads.Real',GnashNames); % all the real load data is consecutive
ll=size(idx,1);
Pstt=idx(1);                            % Pstt & Pstp point to begin and end of real data in GnashArray
Pstp=idx(ll);
nloads=Pstp-Pstt+1;

DateIdx=strmatch('Aux.DayClock',GnashNames); % find the data run of date-time stamps in Matlab format
RDateIdx=strmatch('Aux.DayClock',ReactiveNames);
xx=GnashArray(:,DateIdx);               % the half hourly time stamp (stamped at ??:15 or ??:45 to avoid confusion)
Rxx=ReactiveArray(:,RDateIdx);

for yri=1:NYears                        % yri_idx points to start and end of each year in GnashArray
    yr=yri+YearStt-1;
    yr_idx(yri,1)=min(find(xx>=datenum(yr,1,1,0,0,0)));
    yr_idx(yri,2)=yr_idx(yri,1)+YearLength-1;
    MidYear(yri)=datenum(yr,7,1,0,0,0);
end

% Frequently used constants for the DF & PF calculations

Qnames=regexprep(ReactiveNames,'Reactive','Real'); % used later to find the reactive data corresponding to a real data run
ADMDstt=min(find(xx>=datenum(ADMDYear,1,1,0,0,0)));
ADMDstp=ADMDstt+YearLength-1;
ADMDxx=xx(ADMDstt:ADMDstp);
Rstt=find(Rxx==xx(ADMDstt));
Rstp=find(Rxx==xx(ADMDstp));
clear Rxx
if (isempty(Rstt) | isempty(Rstp))
    disp('Problem with time stamps for reactive data')
end
if (Rstp-Rstt)~=(ADMDstp-ADMDstt)
    disp('Size mismatch problem between reactive and real time stamps')
end
YearsData=GnashArray(ADMDstt:ADMDstp,Pstt:Pstp);
YearsData(isnan(YearsData))=0;
DemTotal=sum(YearsData,2);                   % nans cause a nan sum

% Island Peak, Region Peak, Light Load

% Create Region and Island total

% Check for missing names from the conversion table

for gxp = Pstt:Pstp
    if isempty(strmatch(GnashNames(gxp),ADMDGXPMapping(:,1),'exact'))
        disp(['Missing     ', char(GnashNames(gxp)) , '   from the conversion table']);
    end;
end;

% Island totals

IslandTotal = zeros(ADMDstp-ADMDstt+1,2);
RegionTotal = zeros(ADMDstp-ADMDstt+1,length(Regions));

for islandref = 1:2
    if islandref == 1; Island = 'N'; else Island = 'S'; end;
    ADMDGXPs = strmatch(Island,ADMDGXPMapping(:,3)); % GET ADMD GXPs for island
    for gxp = Pstt:Pstp;
        if ~isempty(strmatch(GnashNames(gxp),ADMDGXPMapping(ADMDGXPs,1),'exact')); % check to see if gxp is in the list for this island
            IslandTotal(:,islandref) = IslandTotal(:,islandref) + YearsData(:, gxp-Pstt+1); % add to accumulated island total if it is
        end;
    end;
end;

% Region totals

for regionref = 1:length(Regions);
    Region = Regions(regionref);
    ADMDGXPs = strmatch(Region,ADMDGXPMapping(:,4)); % GET ADMD GXPs for region
    for gxp = Pstt:Pstp;
        if ~isempty(strmatch(GnashNames(gxp),ADMDGXPMapping(ADMDGXPs,1),'exact')); % check to see if gxp is in the list for this region
            RegionTotal(:,regionref) = RegionTotal(:,regionref) + YearsData(:, gxp-Pstt+1); % add to accumulated region total if it is
        end;
    end;
end;

ESstt=1;        %Extreme Summer
ESstp=max(find(ADMDxx<datenum(ADMDYear,3,1,0,0,0)));  % 1st of March
ExtremeSummer = ESstt:ESstp;          % indices for extreme summer
HHOffset = ESstt-1; % offset for first half hour in the index starting period from the half hour starting midnight.
for i=length(ExtremeSummer):-1:1; if ~(mod(ExtremeSummer(i)+HHOffset,48)>=TimeSummerRatingStarts & mod(ExtremeSummer(i)+HHOffset,48)<=TimeSummerRatingEnds);ExtremeSummer(i)=[];end;end % reduce index to exclude 'night' half hours that have the winter ratings

Sstt1=1;  %Summer - if extreme summer being excluded from summer use Sstt1=ESstp+1;
Sstp1=max(find(ADMDxx<datenum(ADMDYear,5,10,7,0,0)));      %10-May at 07:00:00
Sstt2=max(find(ADMDxx<datenum(ADMDYear,10,20,7,0,0)))+1;   %20-Oct at 07:00:00
Sstp2=size(ADMDxx,1);
Summer=[Sstt1:Sstp1,Sstt2:Sstp2];   % indices for summer
for i=length(Summer):-1:1; if ~(mod(Summer(i)+HHOffset,48)>=TimeSummerRatingStarts & mod(Summer(i)+HHOffset,48)<=TimeSummerRatingEnds);Summer(i)=[];end;end

Wstt=Sstp1+1;   %Winter
Wstp=Sstt2-1;
Winter=Wstt:Wstp;                   % indices for winter
for i=Sstp1:-1:ESstt;                  % plus the night time half hours for summer (done this way for graphing asthetics - keeps the series in order)
    if ~(mod(i+HHOffset,48)>=TimeSummerRatingStarts & mod(i+HHOffset,48)<=TimeSummerRatingEnds);Winter=[i Winter];end;
end;
for i=Sstt2:Sstp2;
    if ~(mod(i+HHOffset,48)>=TimeSummerRatingStarts & mod(i+HHOffset,48)<=TimeSummerRatingEnds);Winter=[Winter i];end;
end;
%
FYstt=1;
FYstp=size(ADMDxx,1);
FullYear = FYstt:FYstp;

% Initialise variables
xxpk=zeros(Npts*NYears,1);              %these two vars to hold one year of data for a GXP
GXPpk=xxpk;

% Total NZ

% Extreme Summer, Summer,Winter and Full Year peak indices
ESTotal=DemTotal(ExtremeSummer);
ESxx=ADMDxx(ExtremeSummer);
[ESTotSort,iiESTotSort]=sort(ESTotal,'descend');
ESPeaksIdx=iiESTotSort(1:ESpts);
ESPeaks = ESTotSort(1:ESpts);
ESMeanPeak = mean(ESPeaks);

STotal=DemTotal(Summer);
Sxx=ADMDxx(Summer);
[STotSort,iiSTotSort]=sort(STotal,'descend');
SPeaksIdx=iiSTotSort(1:Spts);
SPeaks = STotSort(1:Spts);
SMeanPeak = mean(SPeaks);

WTotal=DemTotal(Winter);
Wxx=ADMDxx(Winter);
[WTotSort,iiWTotSort]=sort(WTotal,'descend');
WPeaksIdx=iiWTotSort(1:Wpts);
WPeaks = WTotSort(1:Wpts);
WMeanPeak = mean(WPeaks);

FYTotal = DemTotal(FullYear);
FYxx=ADMDxx(FullYear);
[FYTotSort,iiFYTotSort]=sort(FYTotal,'descend');
FYPeaksIdx=iiFYTotSort(1:Npts);
FYPeaks = FYTotSort(1:Npts);
FYMeanPeak = mean(FYPeaks);

% Index the peaks for the Region and Island Totals

% Extreme Summer

ESIslandTotal = IslandTotal(ExtremeSummer,:); %
ESIslandTotalsSorted = zeros(size(ESIslandTotal,1),2);
ESIslandTotalsIndex = zeros(size(ESIslandTotal,1),2);
for islandref = 1:2;
    [ESIslandTotalsSorted(:,islandref),ESIslandTotalsIndex(:,islandref)] = sort(ESIslandTotal(:,islandref),'descend');
end;
ESIslandPeaksIdx = ESIslandTotalsIndex(1:ESpts,:);
ESIslandPeaks = ESIslandTotalsSorted(1:ESpts,:);
ESIslandMeanPeak = mean(ESIslandPeaks);
ESIslandLightLoadIdx = ESIslandTotalsIndex(size(ESIslandTotal,1)-ESpts+1:size(ESIslandTotal,1),:);
ESIslandLightLoads = ESIslandTotalsSorted(size(ESIslandTotal,1)-ESpts+1:size(ESIslandTotal,1),:);
ESIslandMeanLightLoad = mean(ESIslandLightLoads);

ESRegionTotal = RegionTotal(ExtremeSummer,:);
ESRegionTotalsSorted = zeros(size(ESRegionTotal,1),2);
ESRegionTotalsIndex = zeros(size(ESRegionTotal,1),2);
for regionref = 1:13;
    [ESRegionTotalsSorted(:,regionref),ESRegionTotalsIndex(:,regionref)] = sort(ESRegionTotal(:,regionref),'descend');
end;
ESRegionPeaksIdx = ESRegionTotalsIndex(1:ESpts,:);
ESRegionPeaks = ESRegionTotalsSorted(1:ESpts,:);
ESRegionMeanPeak = mean(ESRegionPeaks);

% Summer

SIslandTotal = IslandTotal(Summer,:);
SIslandTotalsSorted = zeros(size(SIslandTotal,1),2);
SIslandTotalsIndex = zeros(size(SIslandTotal,1),2);
for islandref = 1:2;
    [SIslandTotalsSorted(:,islandref),SIslandTotalsIndex(:,islandref)] = sort(SIslandTotal(:,islandref),'descend');
end;
SIslandPeaksIdx = SIslandTotalsIndex(1:Spts,:);
SIslandPeaks = SIslandTotalsSorted(1:Spts,:);
SIslandMeanPeak = mean(SIslandPeaks);
SIslandLightLoadIdx = SIslandTotalsIndex(size(SIslandTotal,1)-Spts+1:size(SIslandTotal,1),:);
SIslandLightLoads = SIslandTotalsSorted(size(SIslandTotal,1)-Spts+1:size(SIslandTotal,1),:);
SIslandMeanLightLoad = mean(SIslandLightLoads);

SRegionTotal = RegionTotal(Summer,:);
SRegionTotalsSorted = zeros(size(SRegionTotal,1),2);
SRegionTotalsIndex = zeros(size(SRegionTotal,1),2);
for regionref = 1:13;
    [SRegionTotalsSorted(:,regionref),SRegionTotalsIndex(:,regionref)] = sort(SRegionTotal(:,regionref),'descend');
end;
SRegionPeaksIdx = SRegionTotalsIndex(1:Spts,:);
SRegionPeaks = SRegionTotalsSorted(1:Spts,:);
SRegionMeanPeak = mean(SRegionPeaks);

% Winter

WIslandTotal = IslandTotal(Winter,:);
WIslandTotalsSorted = zeros(size(WIslandTotal,1),2);
WIslandTotalsIndex = zeros(size(WIslandTotal,1),2);
for islandref = 1:2;
    [WIslandTotalsSorted(:,islandref),WIslandTotalsIndex(:,islandref)] = sort(WIslandTotal(:,islandref),'descend');
end;
WIslandPeaksIdx = WIslandTotalsIndex(1:Wpts,:);
WIslandPeaks = WIslandTotalsSorted(1:Wpts,:);
WIslandMeanPeak = mean(WIslandPeaks);
WIslandLightLoadIdx = WIslandTotalsIndex(size(WIslandTotal,1)-Wpts+1:size(WIslandTotal,1),:);
WIslandLightLoads = WIslandTotalsSorted(size(WIslandTotal,1)-Wpts+1:size(WIslandTotal,1),:);
WIslandMeanLightLoad = mean(WIslandLightLoads);

WRegionTotal = RegionTotal(Winter,:);
WRegionTotalsSorted = zeros(size(WRegionTotal,1),2);
WRegionTotalsIndex = zeros(size(WRegionTotal,1),2);
for regionref = 1:13;
    [WRegionTotalsSorted(:,regionref),WRegionTotalsIndex(:,regionref)] = sort(WRegionTotal(:,regionref),'descend');
end;
WRegionPeaksIdx = WRegionTotalsIndex(1:Wpts,:);
WRegionPeaks = WRegionTotalsSorted(1:Wpts,:);
WRegionMeanPeak = mean(WRegionPeaks);

% Full Year

FYIslandTotal = IslandTotal(FullYear,:);
FYIslandTotalsSorted = zeros(size(FYIslandTotal,1),2);
FYIslandTotalsIndex = zeros(size(FYIslandTotal,1),2);
for islandref = 1:2;
    [FYIslandTotalsSorted(:,islandref),FYIslandTotalsIndex(:,islandref)] = sort(FYIslandTotal(:,islandref),'descend');
end;
FYIslandPeaksIdx = FYIslandTotalsIndex(1:Npts,:);
FYIslandPeaks = WIslandTotalsSorted(1:Npts,:);
FYIslandMeanPeak = mean(FYIslandPeaks);
FYIslandLightLoadIdx = FYIslandTotalsIndex(size(FYIslandTotal,1)-Npts+1:size(FYIslandTotal,1),:);
FYIslandLightLoads = FYIslandTotalsSorted(size(FYIslandTotal,1)-Npts+1:size(FYIslandTotal,1),:);
FYIslandMeanLightLoad = mean(FYIslandLightLoads);

FYRegionTotal = RegionTotal(FullYear,:);
FYRegionTotalsSorted = zeros(size(FYRegionTotal,1),2);
FYRegionTotalsIndex = zeros(size(FYRegionTotal,1),2);
for regionref = 1:13;
    [FYRegionTotalsSorted(:,regionref),FYRegionTotalsIndex(:,regionref)] = sort(FYRegionTotal(:,regionref),'descend');
end;
FYRegionPeaksIdx = FYRegionTotalsIndex(1:Npts,:);
FYRegionPeaks = FYRegionTotalsSorted(1:Npts,:);
FYRegionMeanPeak = mean(FYRegionPeaks);

% Plot demand totals with peaks identified

if DoRegionGraphs == 1

    %National Total

    figure(10)
    clf
    hold on
    plot(ESxx,ESTotal,'b')
    plot(ESxx(ESPeaksIdx),ESTotal(ESPeaksIdx),'b*')
    plot(Sxx,STotal,'m')
    plot(Sxx(SPeaksIdx),STotal(SPeaksIdx),'m*')
    plot(Wxx,WTotal,'r')
    plot(Wxx(WPeaksIdx),WTotal(WPeaksIdx),'r*')
    title('Total New Zealand');
    hold off
    datetick('x')
    grid on
    axis([xx(ADMDstt) xx(ADMDstp) 0 max(DemTotal).*1.2])

    % Island Graphs

    figure(15)
    for islandref = 1:2
        clf
        hold on
        plot(ESxx,ESIslandTotal(:,islandref),'b')
        plot(ESxx(ESIslandPeaksIdx(:,islandref)),ESIslandTotal(ESIslandPeaksIdx(:,islandref),islandref),'b*')
        plot(Sxx,SIslandTotal(:,islandref),'m')
        plot(Sxx(SIslandPeaksIdx(:,islandref)),SIslandTotal(SIslandPeaksIdx(:,islandref),islandref),'m*')
        plot(Wxx,WIslandTotal(:,islandref),'r')
        plot(Wxx(WIslandPeaksIdx(:,islandref)),WIslandTotal(WIslandPeaksIdx(:,islandref),islandref),'r*')
        if islandref == 1; title('Total North Island'); else title('Total South Island');end
        hold off
        datetick('x')
        grid on
        axis([xx(ADMDstt) xx(ADMDstp) 0 max(WIslandTotal(:,islandref)).*1.2])
        if StepThroughGraphs == 1; pause; end;
        if WriteGraphsToFile == 1;
            if islandref == 1; grfilename = 'North';else grfilename = 'South'; end;
            saveas(gcf,[graphdir '\Island_' grfilename '.png'],'png');
        end;
    end

    % Region Graphs

    figure(20)
    for regionref = 1:13;
        clf
        hold on
        plot(ESxx,ESRegionTotal(:,regionref),'b')
        plot(ESxx(ESRegionPeaksIdx(:,regionref)),ESRegionTotal(ESRegionPeaksIdx(:,regionref),regionref),'b*')
        plot(Sxx,SRegionTotal(:,regionref),'m')
        plot(Sxx(SRegionPeaksIdx(:,regionref)),SRegionTotal(SRegionPeaksIdx(:,regionref),regionref),'m*')
        plot(Wxx,WRegionTotal(:,regionref),'r')
        plot(Wxx(WRegionPeaksIdx(:,regionref)),WRegionTotal(WRegionPeaksIdx(:,regionref),regionref),'r*')
        title(['Total' cellstr(Regions(regionref))]);
        hold off
        datetick('x')
        grid on
        axis([xx(ADMDstt) xx(ADMDstp) 0 max(WRegionTotal(:,regionref)).*1.2])
        if StepThroughGraphs == 1; pause; end;
        if WriteGraphsToFile == 1;
            grfilename = char(Regions(regionref));
            grfilename(find(grfilename=='/')) = '_';
            grfilename(find(grfilename==' ')) = '_';
            saveas(gcf,[graphdir '\Region_' grfilename '.png'],'png');
        end;
    end
end;

% Process one GXP at a time, forming a projection for the Base Year, and
% drawing a graph for visual checking

pd=cd;
if DoGXPGraphs == 1
    figure(1)
    clf
    cd figures
end;
for ii=Pstt:Pstp
    nn=ii-Pstt+1;
    %  find the corresponding entry in the projection control file
    idx=strmatch(GnashNames(ii),PTnames,'exact');
    ll=size(idx,1);
    jj=idx(1);
    if ll>1                             %there are some duplicate names...
        PTnames(idx(1))={'zzzzz'};      %patch so each match is used once only
    end
    
    % Extract yearly data, pkstt,pkstp point to growing list of yearly
    % data. Some years are ignored so the data list for each GXP is of
    % variable length. Two projections are done, one based on groups of
    % NPts for every year to be included (using xxpk and GXPpk), and
    % one based on a single value per year included (based on MeanPeak and
    % MeanPeakYr). The fine resolution one is only for plotting. Each
    % projection can be either a mean of the mpeaks over all specified
    % years, or a linear fit through the peaks in the specified years...
    
    pkstt=1-Npts; MnPkI=0;
    for yri=1:NYears
        if doMx(jj,yri) | strcmp(PType(jj),'fixed')
            pkstt=pkstt+Npts; pkstp=pkstt+Npts-1; MnPkI=MnPkI+1; %MnPkI points to list of yearly totals
            %
            GXPyr=GnashArray(yr_idx(yri,1):yr_idx(yri,2),ii);
            xxyr=xx(yr_idx(yri,1):yr_idx(yri,2));
            [GXPyrSort,iiyrSort]=sort(GXPyr,'descend');
            %
            xxpk(pkstt:pkstp)=xxyr(iiyrSort(1:Npts));
            GXPpk(pkstt:pkstp)=GXPyrSort(1:Npts);
            %
            MeanPeak(MnPkI)=mean(GXPyrSort(1:Npts));
            MeanPeakYr(MnPkI)=YearList(yri);
        end
    end
    
    FitDates(1)=min(xxpk(1:pkstp));                 %for plotting the fine resolution projection
    FitDates(2)=datenum(BaseYear+1,1,1,0,0,0);
    
    switch char(PType(jj))      %three types of projection
        case 'trend'            %fit a linear trend through the peaks
            pp=polyfit(MeanPeakYr(1:MnPkI),MeanPeak(1:MnPkI),1);
            ppsmall=polyfit(xxpk(1:pkstp),GXPpk(1:pkstp),1);
            FitPoints=polyval(ppsmall,FitDates);
            BasePeak(nn)=polyval(pp,BaseYear);
            AnyTimePeak(nn)=polyval(pp,ADMDYear);
        case 'mean'             %the projection is just the mean
            BasePeak(nn)=mean(MeanPeak(1:MnPkI));
            AnyTimePeak(nn)=BasePeak(nn);
            FitPoints(1:2)=mean(GXPpk(1:pkstp));
        case 'fixed'
            BasePeak(nn)=PFixedMW(jj).*1000; % need to convert to KW from MW in table
            AnyTimePeak(nn)=BasePeak(nn);
            FitPoints(1:2) = BasePeak(nn);
        otherwise
            disp('unknown projection type')
    end
    
    % diversity calculation
    
    GXPdemand=GnashArray(ADMDstt:ADMDstp,ii);
    RR=strmatch(GnashNames(ii),Qnames,'exact');
    ll=size(RR,1);
    Ridx=RR(1);
    if ll>1
        Qnames(Ridx)={'zzzzzzz'}; %patch for multiple occurences of same name (Gnash should have produced them in the same order)
    end
    Qdemand=ReactiveArray(Rstt:Rstp,Ridx);

    % Peak demand for ADMD year

    GXPDemandSorted= sort(YearsData(:,nn),'descend');
    GXPPeakDemand(nn) = mean(GXPDemandSorted(1:Npts));

    % Get island and region for the GXP

    if char(ADMDGXPMapping(strmatch(GnashNames(ii),ADMDGXPMapping(:,1),'exact'),3))==char('N');
        islandref = 1;
    else
        islandref = 2;
    end;

    regionref = strmatch(ADMDGXPMapping(strmatch(GnashNames(ii),ADMDGXPMapping(:,1),'exact'),4),Regions);

    % Power factors negative when leading
    
    % Extreme Summer

    ESdemand=GXPdemand(ExtremeSummer);
    ESreactive=Qdemand(ExtremeSummer);
    ESCoincidentP=mean(ESdemand(ESPeaksIdx));
    ESCoincidentQ=mean(ESreactive(ESPeaksIdx));
    ESPangle(nn)=atan2(ESCoincidentQ,ESCoincidentP);
    ESPF(nn)=cos(ESPangle(nn)).*sign(ESPangle(nn));
    ESDF(nn)=ESCoincidentP./GXPPeakDemand(nn);

    ESCoincidentIslandP=mean(ESdemand(ESIslandPeaksIdx(:,islandref)));
    ESCoincidentRegionP=mean(ESdemand(ESRegionPeaksIdx(:,regionref)));
    ESCoincidentIslandLightLoadP=mean(ESdemand(ESIslandLightLoadIdx(:,islandref)));
    ESCoincidentIslandQ=mean(ESreactive(ESIslandPeaksIdx(:,islandref)));
    ESCoincidentRegionQ=mean(ESreactive(ESRegionPeaksIdx(:,regionref)));
    ESCoincidentIslandLightLoadQ=mean(ESreactive(ESIslandLightLoadIdx(:,islandref)));

    ESIslandPangle(nn) = atan2(ESCoincidentIslandQ,ESCoincidentIslandP);
    ESRegionPangle(nn) = atan2(ESCoincidentRegionQ,ESCoincidentRegionP);
    ESIslandLightLoadPangle(nn) = atan2(ESCoincidentIslandLightLoadQ,ESCoincidentIslandLightLoadP);
    ESIslandPF(nn) = cos(ESIslandPangle(nn)).*sign(ESIslandPangle(nn));
    ESIslandDF(nn) = ESCoincidentIslandP./GXPPeakDemand(nn);
    ESRegionPF(nn) = cos(ESRegionPangle(nn)).*sign(ESRegionPangle(nn));
    ESRegionDF(nn) = ESCoincidentRegionP./GXPPeakDemand(nn);
    ESIslandLightLoadPF(nn) = cos(ESIslandLightLoadPangle(nn)).*sign(ESIslandLightLoadPangle(nn));
    ESIslandLightLoadDF(nn) = ESCoincidentIslandLightLoadP./GXPPeakDemand(nn);

    % Summer

    Sdemand=GXPdemand(Summer);
    Sreactive=Qdemand(Summer);
    SCoincidentP=mean(Sdemand(SPeaksIdx));
    SCoincidentQ=mean(Sreactive(SPeaksIdx));
    SPangle(nn)=atan2(SCoincidentQ,SCoincidentP);
    SPF(nn)=cos(SPangle(nn)).*sign(SPangle(nn));
    SDF(nn)=SCoincidentP./GXPPeakDemand(nn);

    SCoincidentIslandP=mean(Sdemand(SIslandPeaksIdx(:,islandref)));
    SCoincidentRegionP=mean(Sdemand(SRegionPeaksIdx(:,regionref)));
    SCoincidentIslandLightLoadP=mean(Sdemand(SIslandLightLoadIdx(:,islandref)));
    SCoincidentIslandQ=mean(Sreactive(SIslandPeaksIdx(:,islandref)));
    SCoincidentRegionQ=mean(Sreactive(SRegionPeaksIdx(:,regionref)));
    SCoincidentIslandLightLoadQ=mean(Sreactive(SIslandLightLoadIdx(:,islandref)));

    SIslandPangle(nn) = atan2(SCoincidentIslandQ,SCoincidentIslandP);
    SRegionPangle(nn) = atan2(SCoincidentRegionQ,SCoincidentRegionP);
    SIslandLightLoadPangle(nn) = atan2(SCoincidentIslandLightLoadQ,SCoincidentIslandLightLoadP);
    SIslandPF(nn) = cos(SIslandPangle(nn)).*sign(SIslandPangle(nn));
    SIslandDF(nn) = SCoincidentIslandP./GXPPeakDemand(nn);
    SRegionPF(nn) = cos(SRegionPangle(nn)).*sign(SRegionPangle(nn));
    SRegionDF(nn) = SCoincidentRegionP./GXPPeakDemand(nn);
    SIslandLightLoadPF(nn) = cos(SIslandLightLoadPangle(nn)).*sign(SIslandLightLoadPangle(nn));
    SIslandLightLoadDF(nn) = SCoincidentIslandLightLoadP./GXPPeakDemand(nn);

    % Winter

    Wdemand=GXPdemand(Winter);
    Wreactive=Qdemand(Winter);
    WCoincidentP=mean(Wdemand(WPeaksIdx));
    WCoincidentQ=mean(Wreactive(WPeaksIdx));
    WPangle(nn)=atan2(WCoincidentQ,WCoincidentP);
    WPF(nn)=cos(WPangle(nn)).*sign(WPangle(nn));
    WDF(nn)=WCoincidentP./GXPPeakDemand(nn);

    WCoincidentIslandP=mean(Wdemand(WIslandPeaksIdx(:,islandref)));
    WCoincidentRegionP=mean(Wdemand(WRegionPeaksIdx(:,regionref)));
    WCoincidentIslandLightLoadP=mean(Wdemand(WIslandLightLoadIdx(:,islandref)));
    WCoincidentIslandQ=mean(Wreactive(WIslandPeaksIdx(:,islandref)));
    WCoincidentRegionQ=mean(Wreactive(WRegionPeaksIdx(:,regionref)));
    WCoincidentIslandLightLoadQ=mean(Wreactive(WIslandLightLoadIdx(:,islandref)));

    WIslandPangle(nn) = atan2(WCoincidentIslandQ,WCoincidentIslandP);
    WRegionPangle(nn) = atan2(WCoincidentRegionQ,WCoincidentRegionP);
    WIslandLightLoadPangle(nn) = atan2(WCoincidentIslandLightLoadQ,WCoincidentIslandLightLoadP);
    WIslandPF(nn) = cos(WIslandPangle(nn)).*sign(WIslandPangle(nn));
    WIslandDF(nn) = WCoincidentIslandP./GXPPeakDemand(nn);
    WRegionPF(nn) = cos(WRegionPangle(nn)).*sign(WRegionPangle(nn));
    WRegionDF(nn) = WCoincidentRegionP./GXPPeakDemand(nn);
    WIslandLightLoadPF(nn) = cos(WIslandLightLoadPangle(nn)).*sign(WIslandLightLoadPangle(nn));
    WIslandLightLoadDF(nn) = WCoincidentIslandLightLoadP./GXPPeakDemand(nn);

    % Full Year

    FYdemand=GXPdemand(FullYear);
    FYreactive=Qdemand(FullYear);
    FYCoincidentP=mean(FYdemand(FYPeaksIdx));
    FYCoincidentQ=mean(FYreactive(FYPeaksIdx));
    FYPangle(nn)=atan2(FYCoincidentQ,FYCoincidentP);
    FYPF(nn)=cos(FYPangle(nn)).*sign(FYPangle(nn));
    FYDF(nn)=FYCoincidentP./GXPPeakDemand(nn);

    FYCoincidentIslandP=mean(FYdemand(FYIslandPeaksIdx(:,islandref)));
    FYCoincidentRegionP=mean(FYdemand(FYRegionPeaksIdx(:,regionref)));
    FYCoincidentIslandLightLoadP=mean(FYdemand(FYIslandLightLoadIdx(:,islandref)));
    FYCoincidentIslandQ=mean(FYreactive(FYIslandPeaksIdx(:,islandref)));
    FYCoincidentRegionQ=mean(FYreactive(FYRegionPeaksIdx(:,regionref)));
    FYCoincidentIslandLightLoadQ=mean(FYreactive(FYIslandLightLoadIdx(:,islandref)));

    FYIslandPangle(nn) = atan2(FYCoincidentIslandQ,FYCoincidentIslandP);
    FYRegionPangle(nn) = atan2(FYCoincidentRegionQ,FYCoincidentRegionP);
    FYIslandLightLoadPangle(nn) = atan2(FYCoincidentIslandLightLoadQ,FYCoincidentIslandLightLoadP);
    FYIslandPF(nn) = cos(FYIslandPangle(nn)).*sign(FYIslandPangle(nn));
    FYIslandDF(nn) = FYCoincidentIslandP./GXPPeakDemand(nn);
    FYRegionPF(nn) = cos(FYRegionPangle(nn)).*sign(FYRegionPangle(nn));
    FYRegionDF(nn) = FYCoincidentRegionP./GXPPeakDemand(nn);
    FYIslandLightLoadPF(nn) = cos(FYIslandLightLoadPangle(nn)).*sign(FYIslandLightLoadPangle(nn));
    FYIslandLightLoadDF(nn) = FYCoincidentIslandLightLoadP./GXPPeakDemand(nn);
    
    % graph results for checking
    if DoGXPGraphs == 1
        figure(1)
        clf
        hold on
        plot(FitDates,FitPoints,'m')        %the trend through individual half hour peaks
        plot(xx,GnashArray(:,ii));          %the actual data
        plot(xxpk(1:pkstp),GXPpk(1:pkstp),'m.') %identify the peaks used in the analysis
        plot([FirstDate, LastDate],[BasePeak(nn), BasePeak(nn)],'r','LineWidth',2) %the answer...
        for kk=1:MnPkI %put a staircase through the annual means
            plot([datenum(MeanPeakYr(kk),1,1,0,0,0),datenum(MeanPeakYr(kk)+1,1,1,0,0,0) ],[MeanPeak(kk),MeanPeak(kk)],'r:','LineWidth',2)
        end
        % graph control
        datetick('x')
        grid on
        MaxY=max(GnashArray(:,ii));
        if ~isnan(MaxY) & ~(MaxY==0)
            axis([FirstDate LastDate 0 MaxY])
        end
        title(char(GnashNames(ii)));
        xlabel('Year')
        ylabel('Demand (kW)')
        hold off
        pgraph = gcf; 

        figure(3)
        clf
        subplot(211)
        hold on
        plot(ESxx,ESdemand,'b')
        plot(ESxx(ESPeaksIdx),ESdemand(ESPeaksIdx),'k.')
        plot(Sxx,Sdemand,'m')
        plot(Sxx(SPeaksIdx),Sdemand(SPeaksIdx),'k.')
        plot(Wxx,Wdemand,'r')
        plot(Wxx(WPeaksIdx),Wdemand(WPeaksIdx),'k.')
        plot([xx(ADMDstt) xx(ADMDstp)],[AnyTimePeak(nn), AnyTimePeak(nn)],'r','LineWidth',2)
        plot(ESxx,ones(size(ESxx)).*ESCoincidentP,'k.','MarkerSize',2)
        plot(Sxx,ones(size(Sxx)).*SCoincidentP,'k.','MarkerSize',2)
        plot(Wxx,ones(size(Wxx)).*WCoincidentP,'k.','MarkerSize',2)
        hold off
        datetick('x')
        grid on
        MaxY=max(GXPdemand);
        if ~isnan(MaxY) & ~(MaxY==0)
            axis([xx(ADMDstt) xx(ADMDstp) 0 MaxY])
        end
        title(char(GnashNames(ii)));
        xlabel('Year')
        ylabel('Demand (kW)')
        %
        subplot(212)
        hold on
        plot(ESxx,ESreactive,'b')
        plot(ESxx(ESPeaksIdx),ESreactive(ESPeaksIdx),'k.')
        plot(Sxx,Sreactive,'m')
        plot(Sxx(SPeaksIdx),Sreactive(SPeaksIdx),'k.')
        plot(Wxx,Wreactive,'r')
        plot(Wxx(WPeaksIdx),Wreactive(WPeaksIdx),'k.')
        plot(ESxx,ones(size(ESxx)).*ESCoincidentQ,'k.','MarkerSize',2)
        plot(Sxx,ones(size(Sxx)).*SCoincidentQ,'k.','MarkerSize',2)
        plot(Wxx,ones(size(Wxx)).*WCoincidentQ,'k.','MarkerSize',2)
        %plot(ESxx,ones(size(ESxx)).*ESCoincidentP,'g.','MarkerSize',2)
        %plot(Sxx,ones(size(Sxx)).*SCoincidentP,'g.','MarkerSize',2)
        %plot(Wxx,ones(size(Wxx)).*WCoincidentP,'g.','MarkerSize',2)
        hold off
        datetick('x')
        grid on
        title(char(GnashNames(ii)));
        xlabel('Year')
        ylabel('Demand (kVA)')

        %print('-dmeta',['DF',num2str(2000+nn)]);
        if StepThroughGraphs == 1; pause; end;    
        if WriteGraphsToFile == 1; % just outputting figure 1
            grfilename = char(GnashNames(ii));
            grfilename(find(grfilename=='/')) = '_';
            grfilename(find(grfilename==' ')) = '_';
            saveas(pgraph,[graphdir '\GXP_' grfilename '.png'],'png');
        end;
    end;
end;
cd(pd)

% save data

names2=strrep(GnashNames,'NILoads.Real.','');
ADMDnames=names2(Pstt:Pstp);

save ADMDdata ADMDnames BasePeak ESDF ESPangle ESPF SDF SPangle SPF WDF WPangle WPF FYDF FYPangle FYPF BaseYear ADMDYear...
    ESIslandPF ESIslandDF ESRegionPF ESRegionDF ESIslandLightLoadPF ESIslandLightLoadDF...
    SIslandPF SIslandDF SRegionPF SRegionDF SIslandLightLoadPF SIslandLightLoadDF...
    WIslandPF WIslandDF WRegionPF WRegionDF WIslandLightLoadPF WIslandLightLoadDF ...
    FYIslandPF FYIslandDF FYRegionPF FYRegionDF FYIslandLightLoadPF FYIslandLightLoadDF ...
    NewLoad GXPPeakDemand GXPExistingEmbeddedMW

warning on MATLAB:divideByZero;

disp('Done');
